<?php
require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/../config/Database.php';

$db = Database::getInstance()->getConnection();

// Pagination
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Filter
$actionFilter = $_GET['action'] ?? '';
$adminFilter = $_GET['admin'] ?? '';
$dateFrom = $_GET['date_from'] ?? '';
$dateTo = $_GET['date_to'] ?? '';

// Build query
$query = "SELECT * FROM admin_activities";
$countQuery = "SELECT COUNT(*) as total FROM admin_activities";
$where = [];
$params = [];

if (!empty($actionFilter)) {
    $where[] = "action LIKE :action";
    $params[':action'] = "%$actionFilter%";
}

if (!empty($adminFilter)) {
    $where[] = "(username LIKE :admin OR admin_id = :admin_id)";
    $params[':admin'] = "%$adminFilter%";
    $params[':admin_id'] = is_numeric($adminFilter) ? $adminFilter : 0;
}

if (!empty($dateFrom)) {
    $where[] = "created_at >= :date_from";
    $params[':date_from'] = $dateFrom;
}

if (!empty($dateTo)) {
    $where[] = "created_at <= :date_to";
    $params[':date_to'] = $dateTo . ' 23:59:59';
}

if (!empty($where)) {
    $query .= " WHERE " . implode(" AND ", $where);
    $countQuery .= " WHERE " . implode(" AND ", $where);
}

// Ordering
$query .= " ORDER BY created_at DESC LIMIT :offset, :per_page";

// Get activities
$stmt = $db->prepare($query);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->bindValue(':per_page', $perPage, PDO::PARAM_INT);
$stmt->execute();
$activities = $stmt->fetchAll();

// Get total count
$stmt = $db->prepare($countQuery);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->execute();
$totalActivities = $stmt->fetch()['total'];
$totalPages = ceil($totalActivities / $perPage);

$pageTitle = "Log Aktivitas Admin";

require_once __DIR__ . '/includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <?php require_once __DIR__ . '/includes/sidebar.php'; ?>
        
        <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">Log Aktivitas Admin</h1>
                <div class="btn-toolbar mb-2 mb-md-0">
                    <button type="button" class="btn btn-sm btn-outline-secondary" onclick="window.print()">
                        <i class="fas fa-print"></i> Cetak
                    </button>
                </div>
            </div>
            
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="mb-0">Filter Log</h5>
                </div>
                <div class="card-body">
                    <form method="get" class="row g-3">
                        <div class="col-md-3">
                            <label for="action" class="form-label">Aksi</label>
                            <input type="text" class="form-control" id="action" name="action" value="<?= htmlspecialchars($actionFilter) ?>" placeholder="Cari aksi...">
                        </div>
                        <div class="col-md-3">
                            <label for="admin" class="form-label">Admin</label>
                            <input type="text" class="form-control" id="admin" name="admin" value="<?= htmlspecialchars($adminFilter) ?>" placeholder="Cari admin...">
                        </div>
                        <div class="col-md-3">
                            <label for="date_from" class="form-label">Dari Tanggal</label>
                            <input type="date" class="form-control" id="date_from" name="date_from" value="<?= htmlspecialchars($dateFrom) ?>">
                        </div>
                        <div class="col-md-3">
                            <label for="date_to" class="form-label">Sampai Tanggal</label>
                            <input type="date" class="form-control" id="date_to" name="date_to" value="<?= htmlspecialchars($dateTo) ?>">
                        </div>
                        <div class="col-12">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-filter"></i> Filter
                            </button>
                            <a href="activities.php" class="btn btn-outline-secondary">
                                Reset
                            </a>
                        </div>
                    </form>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Daftar Aktivitas</h5>
                        <small>Total: <?= number_format($totalActivities) ?> aktivitas</small>
                    </div>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-sm table-hover">
                            <thead>
                                <tr>
                                    <th width="150">Waktu</th>
                                    <th width="120">Admin</th>
                                    <th>Aksi</th>
                                    <th width="120">IP Address</th>
                                    <th width="50"></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($activities as $activity): ?>
                                    <tr>
                                        <td><?= date('d M Y H:i', strtotime($activity['created_at'])) ?></td>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <div class="flex-shrink-0">
                                                    <i class="fas fa-user-circle fa-lg"></i>
                                                </div>
                                                <div class="flex-grow-1 ms-2">
                                                    <?= htmlspecialchars($activity['username']) ?>
                                                    <small class="d-block text-muted">ID: <?= $activity['admin_id'] ?></small>
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <strong><?= htmlspecialchars($activity['action']) ?></strong>
                                            <?php if ($activity['details']): ?>
                                                <div class="text-muted small mt-1">
                                                    <?= htmlspecialchars(substr($activity['details'], 0, 100)) ?>
                                                    <?php if (strlen($activity['details']) > 100): ?>...<?php endif; ?>
                                                </div>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= htmlspecialchars($activity['ip_address']) ?></td>
                                        <td class="text-end">
                                            <button class="btn btn-sm btn-outline-primary" data-bs-toggle="modal" data-bs-target="#detailModal<?= $activity['id'] ?>">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                        </td>
                                    </tr>
                                    
                                    <!-- Modal Detail -->
                                    <div class="modal fade" id="detailModal<?= $activity['id'] ?>" tabindex="-1" aria-hidden="true">
                                        <div class="modal-dialog modal-lg">
                                            <div class="modal-content">
                                                <div class="modal-header">
                                                    <h5 class="modal-title">Detail Aktivitas</h5>
                                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                </div>
                                                <div class="modal-body">
                                                    <div class="row mb-3">
                                                        <div class="col-md-4">
                                                            <strong>Waktu:</strong>
                                                            <p><?= date('d M Y H:i:s', strtotime($activity['created_at'])) ?></p>
                                                        </div>
                                                        <div class="col-md-4">
                                                            <strong>Admin:</strong>
                                                            <p><?= htmlspecialchars($activity['username']) ?> (ID: <?= $activity['admin_id'] ?>)</p>
                                                        </div>
                                                        <div class="col-md-4">
                                                            <strong>IP Address:</strong>
                                                            <p><?= htmlspecialchars($activity['ip_address']) ?></p>
                                                        </div>
                                                    </div>
                                                    
                                                    <div class="mb-3">
                                                        <strong>Aksi:</strong>
                                                        <p><?= htmlspecialchars($activity['action']) ?></p>
                                                    </div>
                                                    
                                                    <?php if ($activity['details']): ?>
                                                        <div class="mb-3">
                                                            <strong>Detail:</strong>
                                                            <pre class="bg-light p-3 rounded"><?= htmlspecialchars(json_encode(json_decode($activity['details']), JSON_PRETTY_PRINT)) ?></pre>
                                                        </div>
                                                    <?php endif; ?>
                                                    
                                                    <?php if ($activity['user_agent']): ?>
                                                        <div class="mb-3">
                                                            <strong>User Agent:</strong>
                                                            <p><?= htmlspecialchars($activity['user_agent']) ?></p>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="modal-footer">
                                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <nav aria-label="Page navigation" class="mt-4">
                        <ul class="pagination justify-content-center">
                            <?php if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>">
                                        &laquo; Sebelumnya
                                    </a>
                                </li>
                            <?php endif; ?>
                            
                            <?php 
                            $startPage = max(1, $page - 2);
                            $endPage = min($totalPages, $page + 2);
                            
                            if ($startPage > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => 1])) ?>">1</a>
                                </li>
                                <?php if ($startPage > 2): ?>
                                    <li class="page-item disabled">
                                        <span class="page-link">...</span>
                                    </li>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
                                <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>">
                                        <?= $i ?>
                                    </a>
                                </li>
                            <?php endfor; ?>
                            
                            <?php if ($endPage < $totalPages): ?>
                                <?php if ($endPage < $totalPages - 1): ?>
                                    <li class="page-item disabled">
                                        <span class="page-link">...</span>
                                    </li>
                                <?php endif; ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $totalPages])) ?>">
                                        <?= $totalPages ?>
                                    </a>
                                </li>
                            <?php endif; ?>
                            
                            <?php if ($page < $totalPages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>">
                                        Selanjutnya &raquo;
                                    </a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                </div>
            </div>
        </main>
    </div>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>