<?php
require __DIR__ . '/config/Config.php';
require __DIR__ . '/config/Database.php';

/**
 * Format tanggal dengan bahasa Indonesia
 */
function indonesianDate($timestamp, $showTime = false) {
    $months = [
        'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni',
        'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'
    ];
    
    $date = date('j', $timestamp) . ' ' . $months[date('n', $timestamp) - 1] . ' ' . date('Y', $timestamp);
    
    if ($showTime) {
        $date .= ' pukul ' . date('H:i', $timestamp);
    }
    
    return $date;
}

/**
 * Redirect dengan pesan flash
 */
function redirect($url, $message = null, $type = 'success') {
    if ($message) {
        $_SESSION['flash_message'] = $message;
        $_SESSION['flash_type'] = $type;
    }
    header("Location: $url");
    exit;
}

/**
 * Tampilkan pesan flash
 */
function displayFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $type = $_SESSION['flash_type'] ?? 'success';
        $message = $_SESSION['flash_message'];
        
        echo '<div class="alert alert-' . htmlspecialchars($type) . ' alert-dismissible fade show">';
        echo htmlspecialchars($message);
        echo '<button type="button" class="btn-close" data-bs-dismiss="alert"></button>';
        echo '</div>';
        
        unset($_SESSION['flash_message']);
        unset($_SESSION['flash_type']);
    }
}

/**
 * Validasi input
 */
function validateInput($data, $rules) {
    $errors = [];
    
    foreach ($rules as $field => $rule) {
        $value = $data[$field] ?? null;
        $rules = explode('|', $rule);
        
        foreach ($rules as $singleRule) {
            if ($singleRule === 'required' && empty($value)) {
                $errors[$field][] = 'Field ini wajib diisi';
            }
            
            if ($singleRule === 'email' && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
                $errors[$field][] = 'Format email tidak valid';
            }
            
            if (strpos($singleRule, 'min:') === 0) {
                $min = (int) substr($singleRule, 4);
                if (strlen($value) < $min) {
                    $errors[$field][] = "Minimal $min karakter";
                }
            }
            
            if (strpos($singleRule, 'max:') === 0) {
                $max = (int) substr($singleRule, 4);
                if (strlen($value) > $max) {
                    $errors[$field][] = "Maksimal $max karakter";
                }
            }
        }
    }
    
    return $errors;
}

/**
 * Dapatkan URL dasar
 */
function baseUrl($path = '') {
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || $_SERVER['SERVER_PORT'] == 443 ? 'https://' : 'http://';
    $host = $_SERVER['HTTP_HOST'];
    $base = str_replace(basename($_SERVER['SCRIPT_NAME']), '', $_SERVER['SCRIPT_NAME']);
    
    return $protocol . $host . $base . ltrim($path, '/');
}

/**
 * Generate random string
 */
function randomString($length = 32) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charactersLength = strlen($characters);
    $randomString = '';
    
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, $charactersLength - 1)];
    }
    
    return $randomString;
}

/**
 * Format bytes to human readable
 */
function formatBytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    
    $bytes /= pow(1024, $pow);
    
    return round($bytes, $precision) . ' ' . $units[$pow];
}

/**
 * Get Telegram user profile photo
 */
function getTelegramUserPhoto($userId, $botToken) {
    $apiUrl = "https://api.telegram.org/bot$botToken/getUserProfilePhotos?user_id=$userId";
    
    $response = file_get_contents($apiUrl);
    $data = json_decode($response, true);
    
    if ($data['ok'] && !empty($data['result']['photos'])) {
        $fileId = $data['result']['photos'][0][0]['file_id'];
        return "https://api.telegram.org/bot$botToken/getFile?file_id=$fileId";
    }
    
    return null;
}