<?php
class AdminHandler {
    private $bot;
    private $db;
    
    public function __construct($bot, $db) {
        $this->bot = $bot;
        $this->db = $db;
    }
    
    public function handleAdminCommand($chatId, $userId, $command, $params = []) {
        $adminIds = explode(',', Config::get('TELEGRAM_ADMIN_IDS'));
        
        if (!in_array($userId, $adminIds)) {
            $this->bot->sendMessage($chatId, "⛔ Anda tidak memiliki izin admin.");
            return;
        }
        
        switch ($command) {
            case 'stats':
                $this->sendStats($chatId);
                break;
                
            case 'broadcast':
                $this->startBroadcast($chatId, $userId, $params);
                break;
                
            case 'userinfo':
                $targetUserId = $params[0] ?? null;
                $this->sendUserInfo($chatId, $targetUserId);
                break;
                
            case 'maintenance':
                $mode = $params[0] ?? 'status';
                $this->handleMaintenance($chatId, $mode);
                break;
                
            default:
                $this->bot->sendMessage($chatId, "Perintah admin tidak dikenali.");
        }
    }
    
    private function sendStats($chatId) {
        $stats = $this->getStats();
        
        $message = "📊 <b>Statistik Bot</b>\n\n";
        $message .= "👥 <b>Pengguna:</b>\n";
        $message .= " - Total: {$stats['total_users']}\n";
        $message .= " - Aktif (30 hari): {$stats['active_users']}\n";
        $message .= " - Baru (hari ini): {$stats['new_users_today']}\n\n";
        
        $message .= "📨 <b>Pesan:</b>\n";
        $message .= " - Total: {$stats['total_messages']}\n";
        $message .= " - Hari ini: {$stats['messages_today']}\n";
        $message .= " - Masuk: {$stats['incoming_messages']}\n";
        $message .= " - Keluar: {$stats['outgoing_messages']}\n\n";
        
        $message .= "📂 <b>File:</b>\n";
        $message .= " - Total: {$stats['total_files']}\n";
        $message .= " - Foto: {$stats['photo_files']}\n";
        $message .= " - Dokumen: {$stats['document_files']}\n";
        
        $this->bot->sendMessage($chatId, $message, null, 'HTML');
    }
    
    private function getStats() {
        $stats = [];
        
        // User stats
        $stmt = $this->db->query("SELECT COUNT(*) as count FROM users");
        $stats['total_users'] = $stmt->fetch()['count'];
        
        $stmt = $this->db->query("
            SELECT COUNT(*) as count 
            FROM users 
            WHERE last_activity > DATE_SUB(NOW(), INTERVAL 30 DAY)
        ");
        $stats['active_users'] = $stmt->fetch()['count'];
        
        $stmt = $this->db->query("
            SELECT COUNT(*) as count 
            FROM users 
            WHERE DATE(join_date) = CURDATE()
        ");
        $stats['new_users_today'] = $stmt->fetch()['count'];
        
        // Message stats
        $stmt = $this->db->query("SELECT COUNT(*) as count FROM messages");
        $stats['total_messages'] = $stmt->fetch()['count'];
        
        $stmt = $this->db->query("
            SELECT COUNT(*) as count 
            FROM messages 
            WHERE DATE(created_at) = CURDATE()
        ");
        $stats['messages_today'] = $stmt->fetch()['count'];
        
        $stmt = $this->db->query("
            SELECT 
                SUM(CASE WHEN direction = 'incoming' THEN 1 ELSE 0 END) as incoming,
                SUM(CASE WHEN direction = 'outgoing' THEN 1 ELSE 0 END) as outgoing
            FROM messages
        ");
        $messageTypes = $stmt->fetch();
        $stats['incoming_messages'] = $messageTypes['incoming'];
        $stats['outgoing_messages'] = $messageTypes['outgoing'];
        
        // File stats
        $stmt = $this->db->query("SELECT COUNT(*) as count FROM files");
        $stats['total_files'] = $stmt->fetch()['count'];
        
        $stmt = $this->db->query("
            SELECT COUNT(*) as count 
            FROM files 
            WHERE file_type = 'photo'
        ");
        $stats['photo_files'] = $stmt->fetch()['count'];
        
        $stmt = $this->db->query("
            SELECT COUNT(*) as count 
            FROM files 
            WHERE file_type = 'document'
        ");
        $stats['document_files'] = $stmt->fetch()['count'];
        
        return $stats;
    }
    
    private function startBroadcast($chatId, $userId, $params) {
        $message = implode(' ', $params);
        
        if (empty($message)) {
            $this->bot->sendMessage($chatId, "Silakan masukkan pesan broadcast setelah perintah.\nContoh: /broadcast Halo semua pengguna!");
            return;
        }
        
        // Simpan state broadcast
        $stmt = $this->db->prepare("
            INSERT INTO admin_states 
            (user_id, state, data, created_at) 
            VALUES (:user_id, 'broadcast', :message, NOW())
            ON DUPLICATE KEY UPDATE 
            data = VALUES(data),
            created_at = NOW()
        ");
        $stmt->execute([
            ':user_id' => $userId,
            ':message' => $message
        ]);
        
        // Kirim konfirmasi
        $keyboard = [
            [['text' => '✅ Ya, kirim', 'callback_data' => 'broadcast_confirm']],
            [['text' => '❌ Batal', 'callback_data' => 'broadcast_cancel']]
        ];
        
        $this->bot->sendMessage(
            $chatId,
            "📢 <b>Konfirmasi Broadcast</b>\n\n" . 
            "Anda akan mengirim pesan berikut ke semua pengguna:\n\n" .
            $message . "\n\n" .
            "Konfirmasi?",
            ['inline_keyboard' => $keyboard],
            'HTML'
        );
    }
    
    private function sendUserInfo($chatId, $targetUserId) {
        if (!$targetUserId) {
            $this->bot->sendMessage($chatId, "Silakan sertakan ID pengguna.\nContoh: /userinfo 123456789");
            return;
        }
        
        $user = $this->getUserDetails($targetUserId);
        
        if (!$user) {
            $this->bot->sendMessage($chatId, "Pengguna dengan ID $targetUserId tidak ditemukan.");
            return;
        }
        
        $message = "👤 <b>Informasi Pengguna</b>\n\n";
        $message .= "🆔 ID: {$user['user_id']}\n";
        $message .= "👤 Nama: {$user['first_name']} {$user['last_name']}\n";
        
        if ($user['username']) {
            $message .= "📛 Username: @{$user['username']}\n";
        }
        
        $message .= "🌐 Bahasa: {$user['language_code']}\n";
        $message .= "🤖 Bot: " . ($user['is_bot'] ? 'Ya' : 'Tidak') . "\n";
        $message .= "📅 Bergabung: " . date('d M Y', strtotime($user['join_date'])) . "\n";
        $message .= "⏱ Terakhir aktif: " . date('d M Y H:i', strtotime($user['last_activity'])) . "\n";
        
        $message .= "\n📊 <b>Statistik:</b>\n";
        $message .= "📨 Total pesan: {$user['message_count']}\n";
        $message .= "📂 File dikirim: {$user['file_count']}\n";
        
        $this->bot->sendMessage($chatId, $message, null, 'HTML');
    }
    
    private function getUserDetails($userId) {
        // Get basic user info
        $stmt = $this->db->prepare("
            SELECT * FROM users WHERE user_id = :user_id
        ");
        $stmt->execute([':user_id' => $userId]);
        $user = $stmt->fetch();
        
        if (!$user) {
            return null;
        }
        
        // Get message count
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as count FROM messages WHERE chat_id = :user_id
        ");
        $stmt->execute([':user_id' => $userId]);
        $user['message_count'] = $stmt->fetch()['count'];
        
        // Get file count
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as count FROM files WHERE user_id = :user_id
        ");
        $stmt->execute([':user_id' => $userId]);
        $user['file_count'] = $stmt->fetch()['count'];
        
        return $user;
    }
    
    private function handleMaintenance($chatId, $mode) {
        $currentMode = Config::get('BOT_MAINTENANCE_MODE') === 'true';
        
        if ($mode === 'status') {
            $status = $currentMode ? 'AKTIF' : 'NON-AKTIF';
            $this->bot->sendMessage($chatId, "🛠 Mode maintenance saat ini: $status");
            return;
        }
        
        if ($mode === 'on') {
            if ($currentMode) {
                $this->bot->sendMessage($chatId, "Mode maintenance sudah aktif.");
                return;
            }
            
            Config::set('BOT_MAINTENANCE_MODE', 'true');
            $this->bot->sendMessage($chatId, "🛠 Mode maintenance diaktifkan. Bot akan merespon hanya ke admin.");
        } elseif ($mode === 'off') {
            if (!$currentMode) {
                $this->bot->sendMessage($chatId, "Mode maintenance sudah non-aktif.");
                return;
            }
            
            Config::set('BOT_MAINTENANCE_MODE', 'false');
            $this->bot->sendMessage($chatId, "🛠 Mode maintenance dinonaktifkan. Bot kembali normal.");
        } else {
            $this->bot->sendMessage($chatId, "Penggunaan: /maintenance [on|off|status]");
            return;
        }
        
        // Save config changes
        Config::save();
    }
    
    public function executeBroadcast($userId, $message) {
        try {
            $adminIds = explode(',', Config::get('TELEGRAM_ADMIN_IDS'));
            
            if (!in_array($userId, $adminIds)) {
                return false;
            }
            
            $result = $this->broadcastToUsers($message);
            
            // Log broadcast
            $stmt = $this->db->prepare("
                INSERT INTO broadcasts 
                (admin_id, message, success_count, total_users, sent_at)
                VALUES (:admin_id, :message, :success_count, :total_users, NOW())
            ");
            
            $stmt->execute([
                ':admin_id' => $userId,
                ':message' => $message,
                ':success_count' => $result['success'],
                ':total_users' => $result['total']
            ]);
            
            return $result;
        } catch (Exception $e) {
            error_log("Broadcast error: " . $e->getMessage());
            return false;
        }
    }
    
    private function broadcastToUsers($message) {
        $stmt = $this->db->query("
            SELECT user_id FROM users WHERE is_active = 1
        ");
        $users = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        $success = 0;
        $failures = 0;
        
        foreach ($users as $userId) {
            try {
                $this->bot->sendMessage($userId, $message, null, 'HTML');
                $success++;
            } catch (Exception $e) {
                error_log("Failed to send to $userId: " . $e->getMessage());
                $failures++;
                
                // Deactivate user if message fails
                $this->deactivateUser($userId);
            }
        }
        
        return [
            'success' => $success,
            'failures' => $failures,
            'total' => count($users)
        ];
    }
    
    private function deactivateUser($userId) {
        $stmt = $this->db->prepare("
            UPDATE users SET is_active = 0 WHERE user_id = :user_id
        ");
        $stmt->execute([':user_id' => $userId]);
    }
}