<?php
class CallbackHandler {
    private $bot;
    private $db;
    
    public function __construct($bot, $db) {
        $this->bot = $bot;
        $this->db = $db;
    }
    
    public function handle($chatId, $userId, $callbackData, $messageId, $callback) {
        // Split callback data into parts (e.g., "menu:main:1")
        $parts = explode(':', $callbackData);
        $mainAction = $parts[0];
        
        switch ($mainAction) {
            case 'main_menu':
                $this->showMainMenu($chatId, $messageId);
                break;
                
            case 'info':
                $this->showInfo($chatId, $messageId);
                break;
                
            case 'help':
                $this->showHelp($chatId, $messageId);
                break;
                
            case 'admin_stats':
                $this->showAdminStats($chatId, $userId, $messageId);
                break;
                
            case 'admin_broadcast':
                $this->startBroadcast($chatId, $userId, $messageId);
                break;
                
            default:
                $this->bot->answerCallbackQuery($callback['id'], "Aksi tidak dikenali");
        }
    }
    
    private function showMainMenu($chatId, $messageId) {
        $keyboard = [
            [['text' => '📊 Informasi', 'callback_data' => 'info']],
            [['text' => '📅 Jadwal', 'callback_data' => 'schedule']],
            [['text' => '📩 Hubungi Kami', 'callback_data' => 'contact']],
            [['text' => '⚙️ Pengaturan', 'callback_data' => 'settings']]
        ];
        
        $this->bot->editMessageText(
            $chatId,
            $messageId,
            "📋 *Menu Utama*\nPilih opsi di bawah:",
            $this->bot->buildInlineKeyboard($keyboard)
        );
    }
    
    private function showInfo($chatId, $messageId) {
        $infoText = "ℹ️ *Informasi Bot*\n\n" .
                    "Ini adalah bot Telegram yang dikembangkan dengan PHP.\n" .
                    "Versi: 1.0.0\n" .
                    "Terakhir diperbarui: " . date('d/m/Y');
        
        $keyboard = [
            [['text' => '🔙 Kembali', 'callback_data' => 'main_menu']]
        ];
        
        $this->bot->editMessageText(
            $chatId,
            $messageId,
            $infoText,
            $this->bot->buildInlineKeyboard($keyboard)
        );
    }
    
    private function showHelp($chatId, $messageId) {
        $helpText = "🆘 *Bantuan*\n\n" .
                    "/start - Memulai bot\n" .
                    "/help - Menampilkan pesan bantuan\n" .
                    "/menu - Menampilkan menu utama\n\n" .
                    "Untuk bantuan lebih lanjut, hubungi admin.";
        
        $keyboard = [
            [['text' => '🔙 Kembali', 'callback_data' => 'main_menu']],
            [['text' => '📞 Hubungi Admin', 'url' => 'https://t.me/admin']]
        ];
        
        $this->bot->editMessageText(
            $chatId,
            $messageId,
            $helpText,
            $this->bot->buildInlineKeyboard($keyboard)
        );
    }
    
    private function showAdminStats($chatId, $userId, $messageId) {
        $adminIds = explode(',', Config::get('TELEGRAM_ADMIN_IDS'));
        
        if (!in_array($userId, $adminIds)) {
            $this->bot->answerCallbackQuery($callback['id'], "⛔ Akses ditolak");
            return;
        }
        
        // Get stats from database
        $stmt = $this->db->prepare("SELECT COUNT(*) as user_count FROM users");
        $stmt->execute();
        $userCount = $stmt->fetch()['user_count'];
        
        $stmt = $this->db->prepare("
            SELECT 
                COUNT(*) as message_count,
                SUM(CASE WHEN direction = 'incoming' THEN 1 ELSE 0 END) as incoming,
                SUM(CASE WHEN direction = 'outgoing' THEN 1 ELSE 0 END) as outgoing
            FROM messages
        ");
        $stmt->execute();
        $messageStats = $stmt->fetch();
        
        $statsText = "📊 *Statistik Admin*\n\n" .
                    "👥 Total Pengguna: $userCount\n" .
                    "📨 Total Pesan:\n" .
                    "  - Masuk: {$messageStats['incoming']}\n" .
                    "  - Keluar: {$messageStats['outgoing']}\n" .
                    "  - Total: {$messageStats['message_count']}\n\n" .
                    "🔄 Terakhir diperbarui: " . date('H:i:s');
        
        $keyboard = [
            [['text' => '🔄 Refresh', 'callback_data' => 'admin_stats']],
            [['text' => '🔙 Kembali', 'callback_data' => 'main_menu']]
        ];
        
        $this->bot->editMessageText(
            $chatId,
            $messageId,
            $statsText,
            $this->bot->buildInlineKeyboard($keyboard)
        );
    }
    
    private function startBroadcast($chatId, $userId, $messageId) {
        $adminIds = explode(',', Config::get('TELEGRAM_ADMIN_IDS'));
        
        if (!in_array($userId, $adminIds)) {
            $this->bot->answerCallbackQuery($callback['id'], "⛔ Akses ditolak");
            return;
        }
        
        $this->bot->editMessageText(
            $chatId,
            $messageId,
            "📢 *Broadcast Message*\n\n" .
            "Balas pesan ini dengan teks yang ingin di-broadcast ke semua pengguna. " .
            "Anda bisa menggunakan format HTML untuk pemformatan teks.\n\n" .
            "Contoh:\n" .
            "<b>Judul</b>\n" .
            "<i>Pesan penting</i>\n" .
            "<a href='https://example.com'>Link</a>",
            null,
            'HTML'
        );
        
        // Set broadcast state in database
        $stmt = $this->db->prepare("
            INSERT INTO admin_states 
            (user_id, state, data, created_at) 
            VALUES (:user_id, 'broadcast', :message_id, NOW())
            ON DUPLICATE KEY UPDATE 
            state = VALUES(state), 
            data = VALUES(data), 
            created_at = NOW()
        ");
        $stmt->execute([
            ':user_id' => $userId,
            ':message_id' => $messageId
        ]);
    }
    
    // Tambahkan method untuk menangani callback lainnya sesuai kebutuhan
}