<?php
class FileHandler {
    private $bot;
    private $db;
    
    public function __construct($bot, $db) {
        $this->bot = $bot;
        $this->db = $db;
    }
    
    public function handleFile($chatId, $userId, $fileData, $fileType, $caption = '') {
        try {
            // Simpan informasi file ke database
            $fileId = $fileData['file_id'];
            $fileUniqueId = $fileData['file_unique_id'];
            $fileSize = $fileData['file_size'] ?? 0;
            
            $stmt = $this->db->prepare("
                INSERT INTO files 
                (user_id, file_id, file_unique_id, file_type, file_size, caption, created_at)
                VALUES (:user_id, :file_id, :file_unique_id, :file_type, :file_size, :caption, NOW())
                ON DUPLICATE KEY UPDATE
                file_size = VALUES(file_size),
                caption = VALUES(caption),
                created_at = NOW()
            ");
            
            $stmt->execute([
                ':user_id' => $userId,
                ':file_id' => $fileId,
                ':file_unique_id' => $fileUniqueId,
                ':file_type' => $fileType,
                ':file_size' => $fileSize,
                ':caption' => $caption
            ]);
            
            // Kirim konfirmasi ke pengguna
            $fileTypes = [
                'photo' => 'Foto',
                'document' => 'Dokumen',
                'video' => 'Video',
                'audio' => 'Audio',
                'voice' => 'Pesan suara'
            ];
            
            $typeName = $fileTypes[$fileType] ?? 'File';
            $message = "✅ $typeName berhasil diterima!";
            
            if (!empty($caption)) {
                $message .= "\n\nCaption: $caption";
            }
            
            $this->bot->sendMessage($chatId, $message);
            
            // Jika admin meminta file, kirim notifikasi
            $this->notifyAdmins($userId, $fileType, $caption);
            
            return true;
        } catch (PDOException $e) {
            error_log("Database error: " . $e->getMessage());
            $this->bot->sendMessage($chatId, "Maaf, terjadi kesalahan saat menyimpan file.");
            return false;
        }
    }
    
    public function getFileLink($fileId) {
        try {
            $botToken = Config::get('TELEGRAM_BOT_TOKEN');
            $apiUrl = "https://api.telegram.org/bot$botToken/getFile?file_id=$fileId";
            
            $response = file_get_contents($apiUrl);
            $data = json_decode($response, true);
            
            if ($data['ok'] && isset($data['result']['file_path'])) {
                return "https://api.telegram.org/file/bot$botToken/" . $data['result']['file_path'];
            }
            
            return false;
        } catch (Exception $e) {
            error_log("Error getting file link: " . $e->getMessage());
            return false;
        }
    }
    
    private function notifyAdmins($userId, $fileType, $caption = '') {
        $adminIds = explode(',', Config::get('TELEGRAM_ADMIN_IDS'));
        $user = $this->getUserInfo($userId);
        
        if (!$user || empty($adminIds)) {
            return;
        }
        
        $userName = $user['first_name'] . ' ' . $user['last_name'];
        if ($user['username']) {
            $userName .= " (@{$user['username']})";
        }
        
        $message = "📤 <b>File Baru Diterima</b>\n";
        $message .= "👤 Pengguna: $userName\n";
        $message .= "📁 Jenis: " . ucfirst($fileType) . "\n";
        
        if (!empty($caption)) {
            $message .= "📝 Caption: $caption\n";
        }
        
        foreach ($adminIds as $adminId) {
            try {
                $this->bot->sendMessage($adminId, $message, null, 'HTML');
            } catch (Exception $e) {
                error_log("Failed to notify admin $adminId: " . $e->getMessage());
            }
        }
    }
    
    private function getUserInfo($userId) {
        $stmt = $this->db->prepare("
            SELECT username, first_name, last_name 
            FROM users 
            WHERE user_id = :user_id
        ");
        $stmt->execute([':user_id' => $userId]);
        return $stmt->fetch();
    }
    
    public function handleFileDownloadRequest($chatId, $fileId) {
        try {
            $fileLink = $this->getFileLink($fileId);
            
            if ($fileLink) {
                $this->bot->sendMessage($chatId, "🔗 Download link: $fileLink");
            } else {
                $this->bot->sendMessage($chatId, "Maaf, tidak dapat mendapatkan link download.");
            }
        } catch (Exception $e) {
            error_log("File download error: " . $e->getMessage());
            $this->bot->sendMessage($chatId, "Terjadi kesalahan saat memproses permintaan.");
        }
    }
}