<?php
class MessageHandler {
    private $bot;
    private $db;
    
    public function __construct($bot, $db) {
        $this->bot = $bot;
        $this->db = $db;
    }
    
    public function handleText($chatId, $userId, $text, $message) {
        // Check if this is a reply to a bot message
        if (isset($message['reply_to_message'])) {
            $this->handleReply($chatId, $userId, $text, $message);
            return;
        }
        
        // Default response
        $defaultResponse = Config::get('BOT_DEFAULT_RESPONSE', 'Maaf, saya tidak mengerti pesan Anda.');
        $this->bot->sendMessage($chatId, $defaultResponse);
    }
    
    public function handlePhoto($chatId, $userId, $photo, $caption, $message) {
        $response = "📷 Terima kasih telah mengirim foto!";
        if ($caption) {
            $response .= "\nCaption: $caption";
        }
        
        $this->bot->sendMessage($chatId, $response);
    }
    
    public function handleDocument($chatId, $userId, $document, $caption, $message) {
        $response = "📄 Terima kasih telah mengirim dokumen!";
        if ($caption) {
            $response .= "\nCaption: $caption";
        }
        
        $this->bot->sendMessage($chatId, $response);
    }
    
    public function handleAudio($chatId, $userId, $audio, $caption, $message) {
        $this->bot->sendMessage($chatId, "🎵 Audio diterima. Terima kasih!");
    }
    
    public function handleVideo($chatId, $userId, $video, $caption, $message) {
        $response = "🎥 Video diterima. Terima kasih!";
        if ($caption) {
            $response .= "\nCaption: $caption";
        }
        
        $this->bot->sendMessage($chatId, $response);
    }
    
    public function handleVoice($chatId, $userId, $voice, $message) {
        $this->bot->sendMessage($chatId, "🎤 Pesan suara diterima. Terima kasih!");
    }
    
    public function handleLocation($chatId, $userId, $location, $message) {
        $lat = $location['latitude'];
        $lon = $location['longitude'];
        $mapsUrl = "https://www.google.com/maps?q=$lat,$lon";
        
        $this->bot->sendMessage(
            $chatId,
            "📍 Lokasi diterima. [Lihat di Google Maps]($mapsUrl)",
            null,
            'Markdown'
        );
    }
    
    public function handleContact($chatId, $userId, $contact, $message) {
        $phone = $contact['phone_number'];
        $name = $contact['first_name'];
        if (isset($contact['last_name'])) {
            $name .= ' ' . $contact['last_name'];
        }
        
        $this->bot->sendMessage(
            $chatId,
            "📱 Kontak diterima:\nNama: $name\nTelepon: $phone"
        );
    }
    
    public function handleUnknown($chatId, $userId, $message) {
        $this->bot->sendMessage(
            $chatId,
            "🤔 Saya tidak mengenali jenis pesan ini. Gunakan /help untuk melihat apa yang bisa saya pahami."
        );
    }
    
    private function handleReply($chatId, $userId, $text, $message) {
        $replyTo = $message['reply_to_message'];
        
        // Check if this is a reply to a bot message
        if ($replyTo['from']['is_bot'] && $replyTo['from']['username'] == Config::get('TELEGRAM_BOT_USERNAME')) {
            // Handle specific reply scenarios
            if (strpos($replyTo['text'], 'Masukkan nama Anda') !== false) {
                $this->handleNameInput($chatId, $userId, $text);
            } elseif (strpos($replyTo['text'], 'Masukkan email Anda') !== false) {
                $this->handleEmailInput($chatId, $userId, $text);
            }
        }
    }
    
    private function handleNameInput($chatId, $userId, $name) {
        // Update user name in database
        $stmt = $this->db->prepare("
            UPDATE users SET first_name = :name, updated_at = NOW() 
            WHERE user_id = :user_id
        ");
        $stmt->execute([':name' => $name, ':user_id' => $userId]);
        
        $this->bot->sendMessage(
            $chatId,
            "Terima kasih, $name! Nama Anda telah disimpan."
        );
    }
    
    private function handleEmailInput($chatId, $userId, $email) {
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->bot->sendMessage($chatId, "Format email tidak valid. Silakan coba lagi.");
            return;
        }
        
        // Update user email in database
        $stmt = $this->db->prepare("
            UPDATE users SET email = :email, updated_at = NOW() 
            WHERE user_id = :user_id
        ");
        $stmt->execute([':email' => $email, ':user_id' => $userId]);
        
        $this->bot->sendMessage(
            $chatId,
            "Email $email telah disimpan. Terima kasih!"
        );
    }
}