<?php
require __DIR__ . '/vendor/autoload.php';
require __DIR__ . '/config/Config.php';
require __DIR__ . '/config/Database.php';
require __DIR__ . '/config/TelegramBot.php';

// Autoload handlers
spl_autoload_register(function ($class) {
    $file = __DIR__ . '/handlers/' . $class . '.php';
    if (file_exists($file)) {
        require $file;
    }
});

// Autoload keyboards
spl_autoload_register(function ($class) {
    $file = __DIR__ . '/keyboards/' . $class . '.php';
    if (file_exists($file)) {
        require $file;
    }
});

// Initialize bot
$bot = new TelegramBot();
$db = Database::getInstance()->getConnection();

// Get input data
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    exit('Invalid input');
}

// Log incoming update
try {
    $stmt = $db->prepare("
        INSERT INTO updates 
        (update_id, update_data, created_at) 
        VALUES (:update_id, :update_data, NOW())
    ");
    $stmt->execute([
        ':update_id' => $data['update_id'],
        ':update_data' => $input
    ]);
} catch (PDOException $e) {
    error_log("Failed to log update: " . $e->getMessage());
}

// Handle different types of updates
if (isset($data['message'])) {
    $message = $data['message'];
    $chatId = $message['chat']['id'];
    $userId = $message['from']['id'];
    $messageId = $message['message_id'];
    
    // Save or update user
    $userData = [
        'username' => $message['from']['username'] ?? null,
        'first_name' => $message['from']['first_name'] ?? null,
        'last_name' => $message['from']['last_name'] ?? null,
        'language_code' => $message['from']['language_code'] ?? null,
        'is_bot' => $message['from']['is_bot'] ? 1 : 0,
        'last_activity' => date('Y-m-d H:i:s')
    ];
    
    $existingUser = $bot->getUser($userId);
    
    if (!$existingUser) {
        $userData['user_id'] = $userId;
        $userData['join_date'] = date('Y-m-d H:i:s');
        $userData['is_active'] = 1;
        
        $stmt = $db->prepare("
            INSERT INTO users 
            (user_id, username, first_name, last_name, language_code, is_bot, join_date, last_activity, is_active) 
            VALUES (:user_id, :username, :first_name, :last_name, :language_code, :is_bot, :join_date, :last_activity, :is_active)
        ");
        $stmt->execute($userData);
    } else {
        $bot->updateUser($userId, $userData);
    }
    
    // Log message
    $messageType = 'text';
    $messageText = $message['text'] ?? '';
    
    if (isset($message['photo'])) {
        $messageType = 'photo';
        $messageText = $message['caption'] ?? 'Photo received';
    } elseif (isset($message['document'])) {
        $messageType = 'document';
        $messageText = $message['caption'] ?? 'Document received';
    } elseif (isset($message['audio'])) {
        $messageType = 'audio';
        $messageText = $message['caption'] ?? 'Audio received';
    } elseif (isset($message['video'])) {
        $messageType = 'video';
        $messageText = $message['caption'] ?? 'Video received';
    } elseif (isset($message['voice'])) {
        $messageType = 'voice';
        $messageText = 'Voice message received';
    } elseif (isset($message['location'])) {
        $messageType = 'location';
        $messageText = 'Location shared';
    } elseif (isset($message['contact'])) {
        $messageType = 'contact';
        $messageText = 'Contact shared';
    }
    
    $bot->logMessage($chatId, $messageText, 'incoming', $messageType);
    
    // Check if bot is in maintenance mode
    if (Config::get('BOT_MAINTENANCE_MODE') === 'true') {
        $bot->sendMessage($chatId, "🤖 Bot sedang dalam mode maintenance. Silakan coba lagi nanti.");
        exit;
    }
    
    // Handle commands
    if (isset($message['text']) && strpos($message['text'], '/') === 0) {
        $command = explode(' ', substr($message['text'], 1))[0];
        $command = explode('@', $command)[0]; // Remove bot username if present
        
        $commandHandler = new CommandHandler($bot, $db);
        $commandHandler->handle($chatId, $userId, $command, $message);
        exit;
    }
    
    // Handle different message types
    $messageHandler = new MessageHandler($bot, $db);
    
    if (isset($message['text'])) {
        $messageHandler->handleText($chatId, $userId, $message['text'], $message);
    } elseif (isset($message['photo'])) {
        $messageHandler->handlePhoto($chatId, $userId, $message['photo'], $message['caption'] ?? '', $message);
    } elseif (isset($message['document'])) {
        $messageHandler->handleDocument($chatId, $userId, $message['document'], $message['caption'] ?? '', $message);
    } elseif (isset($message['audio'])) {
        $messageHandler->handleAudio($chatId, $userId, $message['audio'], $message['caption'] ?? '', $message);
    } elseif (isset($message['video'])) {
        $messageHandler->handleVideo($chatId, $userId, $message['video'], $message['caption'] ?? '', $message);
    } elseif (isset($message['voice'])) {
        $messageHandler->handleVoice($chatId, $userId, $message['voice'], $message);
    } elseif (isset($message['location'])) {
        $messageHandler->handleLocation($chatId, $userId, $message['location'], $message);
    } elseif (isset($message['contact'])) {
        $messageHandler->handleContact($chatId, $userId, $message['contact'], $message);
    } else {
        $messageHandler->handleUnknown($chatId, $userId, $message);
    }
    
} elseif (isset($data['callback_query'])) {
    $callback = $data['callback_query'];
    $chatId = $callback['message']['chat']['id'];
    $userId = $callback['from']['id'];
    $callbackData = $callback['data'];
    $messageId = $callback['message']['message_id'];
    
    $callbackHandler = new CallbackHandler($bot, $db);
    $callbackHandler->handle($chatId, $userId, $callbackData, $messageId, $callback);
    
} elseif (isset($data['channel_post'])) {
    // Handle channel posts if needed
    $channelPost = $data['channel_post'];
    $chatId = $channelPost['chat']['id'];
    
    // Log channel post
    $bot->logMessage($chatId, 'Channel post received', 'incoming', 'channel_post');
    
} elseif (isset($data['edited_message'])) {
    // Handle edited messages
    $editedMessage = $data['edited_message'];
    $chatId = $editedMessage['chat']['id'];
    $userId = $editedMessage['from']['id'];
    
    $bot->logMessage($chatId, 'Message edited: ' . ($editedMessage['text'] ?? 'unknown'), 'incoming', 'edited_message');
}

http_response_code(200);
echo 'OK';